<?php

namespace App\Filament\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Invoice;
use Livewire\Component;
use Filament\Forms\Form;
use Filament\Tables\Table;
use Illuminate\Support\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use Filament\Resources\Resource;
use Filament\Forms\Components\View;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Actions\BulkAction;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\Resources\InvoiceResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Resources\InvoiceResource\RelationManagers;
use App\Filament\Resources\InvoiceResource\RelationManagers\InvoiceItemRelationManager;

class InvoiceResource extends Resource
{
    protected static ?string $model = Invoice::class;

    protected static ?string $navigationIcon = 'heroicon-o-currency-dollar';
    protected static ?string $navigationGroup = 'Penjualan & Invoice';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make(3)
                    ->schema([
                        Forms\Components\TextInput::make('kode_invoice')
                            ->label('Kode Invoice')
                            ->readOnly()
                            ->default(function () {
                                // Ambil code terakhir secara numerik
                                $lastCode = Invoice::query()
                                    ->where('kode_invoice', 'like', 'Bakulpulley-%')
                                    ->orderByRaw('CAST(SUBSTRING(kode_invoice, 5) AS UNSIGNED) DESC')
                                    ->value('kode_invoice');

                                $number = 1;

                                if ($lastCode && preg_match('/Bakulpulley-(\d+)/', $lastCode, $matches)) {
                                    $number = (int) $matches[1] + 1;
                                }

                                return 'Bakulpulley-' . str_pad($number, 4, '0', STR_PAD_LEFT);
                            }),
                        Forms\Components\DatePicker::make('tanggal')
                            ->required(),
                        Forms\Components\Select::make('pelanggan_id')
                            ->searchable()
                            ->preload()
                            ->required()
                            ->relationship('pelanggan', 'nama'),
                    ]),

                Forms\Components\Grid::make(12)
                    ->schema([
                        Forms\Components\TextInput::make('total_display')
                            ->label('Total')
                            ->readOnly()
                            ->disabled()
                            ->dehydrated(false) // agar tidak disimpan ke DB
                            ->formatStateUsing(function ($state, $get) {
                                $value = $get('total') ?? 0;
                                return 'Rp ' . number_format($value, 0, ',', '.');
                            })
                            ->columnSpan(10),


                        View::make('components.refresh-button')
                            ->columnSpan(2),
                    ]),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->query(function () {
                return Invoice::orderBy('id', 'DESC');
            })
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('NO')
                    ->rowIndex()
                    ->searchable(),
                Tables\Columns\TextColumn::make('kode_invoice')
                    ->searchable(),
                Tables\Columns\TextColumn::make('pelanggan.nama')
                    ->searchable(),
                Tables\Columns\TextColumn::make('total')
                    ->numeric()
                    ->searchable(),
                Tables\Columns\TextColumn::make('total')
                    ->label('Total')
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.')),
                Tables\Columns\TextColumn::make('tanggal')
                    ->date()
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
                Filter::make('created_from')
                    ->form([
                        DatePicker::make('created_from')
                            ->label('Requested Tanggal Dari'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['created_from'],
                            fn($query, $date) => $query->whereDate('tanggal', '>=', $date)
                        );
                    }),

                Filter::make('created_to')
                    ->form([
                        DatePicker::make('created_to')
                            ->label('Requested Tanggal Sampai'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query->when(
                            $data['created_to'],
                            fn($query, $date) => $query->whereDate('tanggal', '<=', $date)
                        );
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                // Export Laporan Penjualan
                BulkAction::make('exportPdf')
                    ->label('Export Laporan Penjualan')
                    ->color('info')
                    ->icon('heroicon-o-document')
                    ->action(function ($records) {
                        // Load relasi

                        $data = $records->load(['items', 'pelanggan', 'invoiceItem', 'items.product',])->toArray();
                        // dd($data);
                        // Ambil rentang bulan dari data yang di-*bulk*
                        Carbon::setLocale('id');
                        $bulanAwal = Carbon::parse(min(array_column($data, 'tanggal')))->translatedFormat('F Y');
                        $bulanAkhir = Carbon::parse(max(array_column($data, 'tanggal')))->translatedFormat('F Y');

                        // Format judul PDF berdasarkan periode bulan
                        $judulPeriode = ($bulanAwal == $bulanAkhir) ? $bulanAwal : "$bulanAwal - $bulanAkhir";

                        // dd($judulPeriode);
                        // Buat PDF dalam mode landscape
                        // Generate PDF dengan periode bulan
                        $pdf = Pdf::loadView('exports.LaporanPenjualan', [
                            // 'aboutApp' => $aboutApp,
                            'data' => $data,
                            'judulPeriode' => $judulPeriode
                        ])->setPaper('a4', 'landscape');

                        // Nama file berdasarkan periode bulan
                        return response()->streamDownload(
                            fn() => print($pdf->output()),
                            "LaporanPenjualan_{$judulPeriode}.pdf"
                        );
                    }),
                // exportPdfLaris
                BulkAction::make('exportPdfLaris')
                    ->label('Export Laporan Produk Terlaris')
                    ->color('info')
                    ->icon('heroicon-o-document')
                    ->action(function ($records) {
                        // Load relasi

                        $data = $records->load(['items', 'pelanggan', 'invoiceItem', 'items.product',])->toArray();
                        // dd($data);
                        // Ambil rentang bulan dari data yang di-*bulk*
                        Carbon::setLocale('id');
                        $bulanAwal = Carbon::parse(min(array_column($data, 'tanggal')))->translatedFormat('F Y');
                        $bulanAkhir = Carbon::parse(max(array_column($data, 'tanggal')))->translatedFormat('F Y');

                        // Format judul PDF berdasarkan periode bulan
                        $judulPeriode = ($bulanAwal == $bulanAkhir) ? $bulanAwal : "$bulanAwal - $bulanAkhir";

                        // dd($judulPeriode);
                        // Buat PDF dalam mode landscape
                        // Generate PDF dengan periode bulan
                        $pdf = Pdf::loadView('exports.LaporanProdukTerlaris', [
                            // 'aboutApp' => $aboutApp,
                            'data' => $data,
                            'judulPeriode' => $judulPeriode
                        ])->setPaper('a4', 'landscape');

                        // Nama file berdasarkan periode bulan
                        return response()->streamDownload(
                            fn() => print($pdf->output()),
                            "LaporanProdukTerlaris_{$judulPeriode}.pdf"
                        );
                    }),
                // exportLabaRugi
                BulkAction::make('exportLabaRugi')
                    ->label('Export Laba Rugi')
                    ->color('warning')
                    ->icon('heroicon-o-currency-dollar')
                    ->action(function ($records) {
                        $data = $records->load(['items', 'pelanggan', 'items.product'])->toArray();

                        \Carbon\Carbon::setLocale('id');
                        $bulanAwal = \Carbon\Carbon::parse(min(array_column($data, 'tanggal')))->translatedFormat('F Y');
                        $bulanAkhir = \Carbon\Carbon::parse(max(array_column($data, 'tanggal')))->translatedFormat('F Y');
                        $judulPeriode = ($bulanAwal === $bulanAkhir) ? $bulanAwal : "$bulanAwal - $bulanAkhir";

                        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('exports.LaporanLabaRugi', [
                            'data' => $data,
                            'judulPeriode' => $judulPeriode,
                        ])->setPaper('a4', 'landscape');

                        return response()->streamDownload(
                            fn() => print($pdf->output()),
                            "Laporan_Laba_Rugi_{$judulPeriode}.pdf"
                        );
                    }),
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
            InvoiceItemRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListInvoices::route('/'),
            'create' => Pages\CreateInvoice::route('/create'),
            'edit' => Pages\EditInvoice::route('/{record}/edit'),
        ];
    }
}
