<?php

namespace App\Filament\Resources\InvoiceResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class InvoiceItemRelationManager extends RelationManager
{
    protected static string $relationship = 'invoiceItem';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('product_id')
                    ->label('Kode Produk - Nama - Stock')
                    ->relationship('product', 'nama')
                    ->getOptionLabelFromRecordUsing(function ($record) {
                        return "{$record->kode_produk} - {$record->nama} - Stock {$record->stok}";
                    })
                    ->preload()
                    ->searchable()
                    ->required()
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $set) {
                        // Ambil harga_jual dari product dan set ke harga_satuan
                        $product = \App\Models\Product::find($state);
                        if ($product) {
                            $set('harga_satuan', $product->harga_jual);
                            $set('subtotal', $product->harga_jual); // default kuantitas = 1
                        } else {
                            $set('harga_satuan', null);
                            $set('subtotal', null);
                        }
                    }),

                Forms\Components\TextInput::make('kuantitas')
                    ->label('Kuantitas')
                    ->numeric()
                    ->required()
                    ->default(1)
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $get, callable $set) {
                        $harga = $get('harga_satuan') ?? 0;
                        $set('subtotal', $state * $harga);
                    }),

                Forms\Components\Placeholder::make('harga_satuan_display')
                    ->label('Harga Satuan')
                    ->content(function ($get) {
                        $value = $get('harga_satuan') ?? 0;
                        return 'Rp ' . number_format($value, 0, ',', '.');
                    }),

                Forms\Components\Placeholder::make('subtotal_display')
                    ->label('Subtotal')
                    ->content(function ($get) {
                        $value = $get('subtotal') ?? 0;
                        return 'Rp ' . number_format($value, 0, ',', '.');
                    }),
                Forms\Components\Hidden::make('harga_satuan'),
                Forms\Components\Hidden::make('subtotal'),


            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('nama')
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('NO')
                    ->rowIndex()
                    ->searchable(),
                Tables\Columns\TextColumn::make('invoice.kode_invoice')
                    ->label('Kode Invoice'),
                Tables\Columns\TextColumn::make('product.kode_produk'),
                Tables\Columns\TextColumn::make('product.nama'),
                Tables\Columns\TextColumn::make('kuantitas'),
                Tables\Columns\TextColumn::make('harga_satuan')
                    ->label('Harga Satuan')
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.')),

                Tables\Columns\TextColumn::make('subtotal')
                    ->label('Subtotal')
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.')),

            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
    // total
    protected function afterCreate(): void
    {
        $this->getOwnerRecord()->updateTotal();
    }
    protected function afterEdit(): void
    {
        $this->getOwnerRecord()->updateTotal();
    }
    protected function afterDelete(): void
    {
        $this->getOwnerRecord()->updateTotal();
    }
}
