<?php

namespace App\Filament\Resources;

use Filament\Forms;
use Filament\Tables;
use App\Models\Product;
use App\Models\Produksi;
use Filament\Forms\Form;
use Filament\Tables\Table;
use App\Models\ResepProduk;
use Filament\Resources\Resource;
use Illuminate\Database\Eloquent\Builder;
use App\Filament\Resources\ProduksiResource\Pages;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Filament\Resources\ProduksiResource\RelationManagers;
use Filament\Forms\Components\{Card, Select, TextInput, DatePicker, Placeholder, Grid};

class ProduksiResource extends Resource
{
    protected static ?string $model = Produksi::class;

    protected static ?string $navigationIcon = 'heroicon-o-cog-8-tooth';
    protected static ?string $navigationGroup = 'Produksi (HPP Count)';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([

                Card::make([
                    Select::make('product_id')
                        ->label('Produk')
                        ->relationship('product', 'nama')
                        ->preload()
                        ->searchable()
                        ->reactive()
                        ->afterStateUpdated(fn($state, callable $set) => $set('bahan', json_encode(
                            ResepProduk::with('bahanBaku')->where('product_id', $state)->get()
                        ))),

                    TextInput::make('berat_total')->label('Berat Total (gram)')->numeric()->required(),
                    TextInput::make('biaya_bubut')->label('Biaya Bubut')->numeric()->default(0),
                    TextInput::make('biaya_finish')->label('Biaya Finishing')->numeric()->default(0),
                    TextInput::make('jumlah_dibuat')->label('Jumlah Dibuat')->numeric()->required(),
                    DatePicker::make('tanggal')->default(now()),

                    // Bahan baku (readonly preview)
                    Placeholder::make('komposisi_bahan')
                        ->label('Komposisi Bahan Baku')
                        ->content(function ($get) {
                            $bahan = json_decode($get('bahan'), true);
                            if (!$bahan) return 'Silakan pilih produk terlebih dahulu.';
                            return collect($bahan)->map(function ($item) {
                                return "- {$item['bahan_baku']['nama']} ({$item['kuantitas']} {$item['bahan_baku']['satuan']})";
                            })->implode('<br>');
                        })->extraAttributes(['class' => 'text-sm text-gray-700'])->columnSpanFull(),
                ]),
            ]);
    }
    public static function table(Table $table): Table
    {
        return $table
            ->query(function () {
                return Produksi::orderBy('id', 'DESC');
            })
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('NO')
                    ->rowIndex()
                    ->searchable(),
                Tables\Columns\TextColumn::make('product.kode_produk')
                    ->label('Kode')
                    ->sortable(),
                Tables\Columns\TextColumn::make('product.nama')
                    ->label('Produk')
                    ->sortable(),
                Tables\Columns\TextColumn::make('product.harga_jual')
                    ->numeric()
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.'))
                    ->label('Harga Jual')
                    ->sortable(),
                Tables\Columns\TextColumn::make('product.stok')
                    ->label('Stok')
                    ->sortable(),
                Tables\Columns\TextColumn::make('berat_total')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('biaya_bubut')
                    ->numeric()
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.'))
                    ->sortable(),
                Tables\Columns\TextColumn::make('biaya_finish')
                    ->numeric()
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.'))
                    ->sortable(),
                Tables\Columns\TextColumn::make('hpp_per_item')
                    ->numeric()
                    ->formatStateUsing(fn($state) => 'Rp ' . number_format($state, 0, ',', '.'))
                    ->sortable(),
                Tables\Columns\TextColumn::make('jumlah_dibuat')
                    ->numeric()
                    ->sortable(),
                Tables\Columns\TextColumn::make('tanggal')
                    ->label('Tanggal Produksi')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListProduksis::route('/'),
            'create' => Pages\CreateProduksi::route('/create'),
            'edit' => Pages\EditProduksi::route('/{record}/edit'),
        ];
    }
}
